// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/DecayedParticles.hh"

namespace Rivet {


  /// @brief 
  class CLEO_2006_I710864 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(CLEO_2006_I710864);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      _mode = 0;
      if ( getOption("MODE") == "SIGMA" ) _mode = 0;
      else if ( getOption("MODE") == "DECAY" ) _mode = 1;
      // set the PDG code
      _pid = getOption<int>("PID", 9030443);
      // setup for cross sections
      if(_mode==0) {
	// projections
	declare(FinalState(), "FS");
	declare(UnstableParticles(Cuts::pid== 443), "UFS");
	// histograms
	for(unsigned int ix=0;ix<3;++ix)
	  book(_c[ix],"TMP/c_"+toString(ix));
      }
      // setup for decays
      else {
	// projections
	UnstableParticles ufs = UnstableParticles(Cuts::abspid==_pid);
	declare(ufs, "UFS");
	DecayedParticles PSI(ufs);
	PSI.addStable(443);
	declare(PSI, "PSI");
	book(_h,2,1,1);
      }
    }

    void findChildren(const Particle & p,map<long,int> & nRes, int &ncount) {
      for (const Particle &child : p.children()) {
	if(child.children().empty()) {
	  --nRes[child.pid()];
	  --ncount;
	}
	else
	  findChildren(child,nRes,ncount);
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // cross section
      if(_mode==0) {
	const FinalState& fs = apply<FinalState>(event, "FS");
	map<long,int> nCount;
	int ntotal(0);
	for (const Particle& p : fs.particles()) {
	  nCount[p.pid()] += 1;
	  ++ntotal;
	}
	for(const Particle & psi : apply<UnstableParticles>(event, "UFS").particles()) {
	  if(psi.children().empty()) continue;
	  map<long,int> nRes = nCount;
	  int ncount = ntotal;
	  findChildren(psi,nRes,ncount);
	  if(ncount!=2) continue;
	  if(nRes.find(211)!=nRes.end() && nRes.find(-211)!=nRes.end() &&
	     nRes[211]==1 && nRes[-211]==1) {
	    _c[0]->fill();
	    break;
	  }
	  else if(nRes.find(111)!=nRes.end() && nRes[111]==2) {
	    _c[1]->fill();
	    break;
	  }
	  else if(nRes.find(321)!=nRes.end() && nRes.find(-321)!=nRes.end() &&
	     nRes[321]==1 && nRes[-321]==1) {
	    _c[2]->fill();
	    break;
	  }
	}
      }
      // decays
      else {
	static const map<PdgId,unsigned int> & mode = { { 211,1}, {-211,1}, { 443,1}};
	DecayedParticles PSI = apply<DecayedParticles>(event, "PSI");
	for(unsigned int ix=0;ix<PSI.decaying().size();++ix) {
	  if (!PSI.modeMatches(ix,3,mode)) continue;
	  const Particle & pip  = PSI.decayProducts()[ix].at( 211)[0];
	  const Particle & pim  = PSI.decayProducts()[ix].at(-211)[0];
	  double mpipi = (pip.momentum()+pim.momentum()).mass();
	  _h->fill(mpipi);
	}
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      // sigma
      if(_mode==0) {
	double fact = crossSection()/ sumOfWeights() /picobarn; 
	for(unsigned int ix=0;ix<3;++ix) {
	  double sigma = _c[ix]->val()*fact;
	  double error = _c[ix]->err()*fact;
	  Scatter2D temphisto(refData(1, 1, 1+ix));
	  Scatter2DPtr  mult;
	  book(mult, 1, 1, 1+ix);
	  for (size_t b = 0; b < temphisto.numPoints(); b++) {
	    const double x  = temphisto.point(b).x();
	    pair<double,double> ex = temphisto.point(b).xErrs();
	    pair<double,double> ex2 = ex;
	    if(ex2.first ==0.) ex2. first=0.0001;
	    if(ex2.second==0.) ex2.second=0.0001;
	    if (inRange(sqrtS()/GeV, x-ex2.first, x+ex2.second)) {
	      mult->addPoint(x, sigma, ex, make_pair(error,error));
	    }
	    else {
	      mult->addPoint(x, 0., ex, make_pair(0.,.0));
	    }
	  }
	}
      }
      // decay
      else {
	normalize(_h,1.,false);
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    int _mode;
    int _pid;
    Histo1DPtr _h;
    CounterPtr _c[3];
    /// @}


  };


  RIVET_DECLARE_PLUGIN(CLEO_2006_I710864);

}
